function lFig=plotCube(aDataCube,aCipMetaData,aTitle,aPlotType,aPlotPris)
% plotCube Plot the data cube in tiled format
%   Plot the data cube in tiled format with a common y-limit to allow
%   comparisons.  Arguments:
%   * aDataCube(Channel,Range,Pulse)
%   * aCipMetaData structure describing the meta data for the CIP
%   * aTitle title for figure, by convention include the record file name if
%     applicable.
%   * aPlotType -- 1=real/imag data
%                  2=magnitude (dBq)
%                  3=dB
%                  
%   * aFigOrder -- order to print the channels (must be smaller than the
%   first subscript of aDataCube
%
lFig=figure;
lFigOrder = radChan.getFigOrder();
lSrChans = radChan.getSrChans();
lLrChans = radChan.getLrChans();
set(lFig,'position',[40 50 1500 600]);
set(lFig,'Name',aTitle);
tiledlayout(2,4);
% figure out y axis limits
aDataCube = reshape(aDataCube,aCipMetaData.numChannels,[]);
maxValue = zeros(1,aCipMetaData.numChannels);
maxValue(1,lLrChans)=max(max(abs(aDataCube(lLrChans,:))));
maxValue(1,lSrChans)=max(max(abs(aDataCube(lSrChans,:))));
rOrder=10^(floor(log10(max(maxValue))));
plotLimit=ceil(max(maxValue)/rOrder)*rOrder;

% select PRIs to include in the plot...
startIdx = 1;
if ~exist('aPlotPris','var')
    aPlotPris = 3;  
end

startPlot=(startIdx-1)*aCipMetaData.priSize+1;
endPlot=startPlot+(aPlotPris*aCipMetaData.priSize)-1;

for ii=lFigOrder
    ax=nexttile;
    switch aPlotType
        case 'IQ'
            % plot real and imaginary
            plot(real(aDataCube(ii,startPlot:endPlot)),'b');
            hold on;
            plot(imag(aDataCube(ii,startPlot:endPlot)),'r');
            ylim([-plotLimit plotLimit]);
            ylabel('Counts');
            xlabel('Samples');
        case 'Mag'
            % plot magnitude in dBq
            plot(20*log10(abs(aDataCube(ii,startPlot:endPlot))),'b'); 
            ylim([0 20*log10(plotLimit)]);
            ylabel('dBq');
            xlabel('Samples');
        case 'MagDBPeak'
            % plot dbPeak
            plot(20*log10(abs(aDataCube(ii,startPlot:endPlot))/maxValue(ii)),'color',[0.3 0.6 0.4]);
            ylim([-120 5]);
            ylabel('dB');
            xlabel('Samples');
        case 'MagSq'
            % plot normalized MagSquared
            plot(10*log10((aDataCube(ii,startPlot:endPlot))),'color',[0.3 0.4 0.6]);
            ylim([-50 100]);
            ylabel('dB');
            xlabel('Samples');
        case 'DopMaxClut'
            % plot doppler data for the clutter peak (mag squared data)
            % find the peak doppler return...
            aDataCube = reshape(aDataCube,aCipMetaData.numChannels,aCipMetaData.priSize,[]); % undo the reshaping....
            [~,rangeCell]=max(aDataCube(ii,:,81)); %todo fix magic number

            plot(aCipMetaData.dopplerBins_mps,10*log10(squeeze(aDataCube(ii,rangeCell,:))/maxValue(ii)),'color',[0.3 0.4 0.6]);
            ylim([-120 5]);
            ylabel('dB');
            xlim([floor(aCipMetaData.dopplerBins_mps(1)) ceil(aCipMetaData.dopplerBins_mps(end))]);
            xlabel('Dopper Bin (mps)');
        case 'Spectrum'
            % plot spectrum
            sampleRate = aCipMetaData.sampleRate_sps(ii);
            freq=((-sampleRate/2):sampleRate/aCipMetaData.priSize:sampleRate/2)/1e6;
            fdomain=fftshift(fft(aDataCube(ii,1:aCipMetaData.priSize)));
            plot(freq(1:end-1),20*log10(abs(fdomain)));
            xlabel("MHz"),ylabel('dBq');
          
        otherwise
            error('plotCube: aPlotType unknown');

    end
    title(radChan.getString(ii));

    % Create a custom toolbar with no buttons
    tb = axtoolbar(ax,{'datacursor','zoomin','zoomout','restoreview','pan'});
    
    % Create a custom export button.
    btn = axtoolbarbtn(tb, 'push', 'Icon', 'export');
    
    % When the button is pushed, use copyobj to copy the axes into a new
    % figure
    btn.ButtonPushedFcn = @(~,e) copyobj(e.Axes,tiledlayout(figure,'flow'));
end
end

